<?php

namespace Symfony\Component\Console\Input;

/*
 * This file is part of the symfony framework.
 *
 * (c) Fabien Potencier <fabien.potencier@symfony-project.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

/**
 * Represents a command line argument.
 *
 * @package    symfony
 * @subpackage console
 * @author     Fabien Potencier <fabien.potencier@symfony-project.com>
 */
class InputArgument
{
  const REQUIRED = 1;
  const OPTIONAL = 2;
  const IS_ARRAY = 4;

  protected $name;
  protected $mode;
  protected $default;
  protected $description;

  /**
   * Constructor.
   *
   * @param string  $name        The argument name
   * @param integer $mode        The argument mode: self::REQUIRED or self::OPTIONAL
   * @param string  $description A description text
   * @param mixed   $default     The default value (for self::OPTIONAL mode only)
   */
  public function __construct($name, $mode = null, $description = '', $default = null)
  {
    if (null === $mode)
    {
      $mode = self::OPTIONAL;
    }
    else if (is_string($mode) || $mode > 7)
    {
      throw new \InvalidArgumentException(sprintf('Argument mode "%s" is not valid.', $mode));
    }

    $this->name        = $name;
    $this->mode        = $mode;
    $this->description = $description;

    $this->setDefault($default);
  }

  /**
   * Returns the argument name.
   *
   * @return string The argument name
   */
  public function getName()
  {
    return $this->name;
  }

  /**
   * Returns true if the argument is required.
   *
   * @return Boolean true if parameter mode is self::REQUIRED, false otherwise
   */
  public function isRequired()
  {
    return self::REQUIRED === (self::REQUIRED & $this->mode);
  }

  /**
   * Returns true if the argument can take multiple values.
   *
   * @return Boolean true if mode is self::IS_ARRAY, false otherwise
   */
  public function isArray()
  {
    return self::IS_ARRAY === (self::IS_ARRAY & $this->mode);
  }

  /**
   * Sets the default value.
   *
   * @param mixed $default The default value
   */
  public function setDefault($default = null)
  {
    if (self::REQUIRED === $this->mode && null !== $default)
    {
      throw new \LogicException('Cannot set a default value except for Parameter::OPTIONAL mode.');
    }

    if ($this->isArray())
    {
      if (null === $default)
      {
        $default = array();
      }
      else if (!is_array($default))
      {
        throw new \LogicException('A default value for an array argument must be an array.');
      }
    }

    $this->default = $default;
  }

  /**
   * Returns the default value.
   *
   * @return mixed The default value
   */
  public function getDefault()
  {
    return $this->default;
  }

  /**
   * Returns the description text.
   *
   * @return string The description text
   */
  public function getDescription()
  {
    return $this->description;
  }
}
